<?php 

class DeviceRemover {
    /** @var \mysqli */
    private $conn;
    
    /** @var string */
    private $apiUrl;
    
    /** @var string */
    private $apiToken = 'eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9';

    /**
     * @param \mysqli $connection
     * @param string $apiUrl
     */
    public function __construct(\mysqli $connection, string $apiUrl) {
        $this->conn = $connection;
        $this->apiUrl = $apiUrl;
    }

    /**
     * @param string $username
     * @param string $password
     * @param bool $confirmRemove
     * @param int $userId
     * @return string
     * @throws \Exception
     */
    public function execute(string $username, string $password, bool $confirmRemove, int $userId): string {
        $payload = [
            "username" => $username,
            "password" => base64_encode($password),
            "remove" => "true"
        ];

        $response = $this->callApi($payload);
        $responseData = json_decode($response, true);

        // Validate API response structure
        if (!is_array($responseData)) {
            throw new \Exception("Respuesta de la API en formato inválido.");
        }

        if (!isset($responseData['success'])) {
            throw new \Exception("Formato de respuesta inesperado: falta campo 'success'.");
        }

        if (!$responseData['success']) {
            $errorMessage = $responseData['error'] ?? 'Error desconocido de la API';
            throw new \Exception("La API respondió con error: " . $errorMessage);
        }

        // Handle cases where devices might be null, empty array, or missing
        $devices = [];
        if (isset($responseData['devices'])) {
            if (is_array($responseData['devices'])) {
                $devices = $responseData['devices'];
            } else if ($responseData['devices'] === null) {
                $devices = []; // Convert null to empty array
            }
        }

        $this->logToDatabase($userId, $username, $password, $responseData);

        return $this->buildOutput($devices);
    }

    /**
     * @param array $payload
     * @return string
     * @throws \Exception
     */
    private function callApi(array $payload): string {
        $ch = curl_init($this->apiUrl);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($payload),
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Autkm: ' . $this->apiToken,
            ],
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CONNECTTIMEOUT => 10
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);

        if ($curlError) {
            throw new \Exception("Error de conexión: " . $curlError);
        }

        if ($httpCode !== 200) {
            throw new \Exception("La API falló con HTTP status $httpCode");
        }

        // Log the response for debugging
        file_put_contents("device.log", $response . PHP_EOL, FILE_APPEND);
        
        return $response;
    }

    /**
     * @param int $userId
     * @param string $username
     * @param string $password
     * @param array $response
     */
    private function logToDatabase(int $userId, string $username, string $password, array $response): void {
        $stmt = $this->conn->prepare("
            INSERT INTO history_remove (user_id, username, password, response_data, created_at)
            VALUES (?, ?, ?, ?, NOW())
        ");
        
        $json = json_encode($response, JSON_UNESCAPED_UNICODE);
        $stmt->bind_param("isss", $userId, $username, $password, $json);
        $stmt->execute();
        $stmt->close();
    }

    /**
     * @param array $devices
     * @return string
     */
    private function buildOutput(array $devices): string {
        // Handle empty devices case
        if (empty($devices)) {
            return "Sin Dispositivos";
        }

        $output = "";
        $excluded = ['id', 'img'];
        $fieldLabels = [
            'name' => '👤 Nombre',
            'model' => '📱 Modelo',
            'was_removed' => '🟢 Eliminado',
            'battery_status' => '🔋 Estado batería',
            'battery_level' => '🔋 Nivel de batería',
            'location_enabled' => '📍 Localización',
            'lost_mode' => '🛑 Modo perdido',
            'lost_message' => '📢 Mensaje de pérdida',
            'lost_owner' => '👤 Teléfono de dueño',
            'lost_email' => '📧 Correo de pérdida',
            'remote_wipe' => '💬 Mensaje remoto',
            'udid' => '🆔 UDID',
            'serial' => '🔢 Serial',
            'imei' => '📲 IMEI',
            'iOS' => '📱 iOS'
        ];

        foreach ($devices as $device) {
            // Skip non-array items or null values
            if (!is_array($device) || empty($device)) {
                continue;
            }

            $deviceOutput = "";
            $hasData = false;

            foreach ($fieldLabels as $key => $label) {
                if (in_array($key, $excluded, true)) continue;
                
                if (!isset($device[$key]) || $device[$key] === null || $device[$key] === '') {
                    continue;
                }

                $hasData = true;
                
                switch ($key) {
                    case 'was_removed':
                        $deviceOutput .= "$label: " . ($device[$key] ? 'Sí' : 'No') . "\n";
                        break;
                    case 'location_enabled':
                        $deviceOutput .= "$label: " . ($device[$key] ? 'Activada' : 'Desactivada') . "\n";
                        break;
                    case 'lost_mode':
                        $deviceOutput .= "$label: " . strtoupper($device[$key]) . "\n";
                        break;
                    default:
                        $deviceOutput .= "$label: " . $device[$key] . "\n";
                        break;
                }
            }

            if ($hasData) {
                $output .= $deviceOutput . "\n";
            }
        }

        // If no devices had valid data, return default message
        return empty($output) ? "Sin Dispositivos" : $output;
    }
}