<?php
/**
 * 🌐 IP Detector - Detección y Análisis de IPs Optimizado
 * 
 * Detecta IPs y las analiza para identificar patrones sospechosos
 * sin hacer llamadas externas lentas
 */

namespace AntiBotTr;

class IPDetector
{
    private $suspiciousRanges = [
        // Rangos de datacenters conocidos
        [1, 1], [1, 2], [2, 2], [3, 3], [4, 4], [5, 5],
        [8, 8], [9, 9], [11, 11], [12, 12], [13, 13], [14, 14],
        [15, 15], [16, 16], [17, 17], [18, 18], [19, 19], [20, 20],
        [23, 23], [24, 24], [25, 25], [26, 26], [27, 27], [28, 28],
        [29, 29], [30, 30], [31, 31], [32, 32], [33, 33], [34, 34],
        [35, 35], [36, 36], [37, 37], [38, 38], [39, 39], [40, 40],
        [41, 41], [42, 42], [43, 43], [44, 44], [45, 45], [46, 46],
        [47, 47], [48, 48], [49, 49], [50, 50], [51, 51], [52, 52],
        [53, 53], [54, 54], [55, 55], [56, 56], [57, 57], [58, 58],
        [59, 59], [60, 60], [61, 61], [62, 62], [63, 63], [64, 64],
        [65, 65], [66, 66], [67, 67], [68, 68], [69, 69], [70, 70],
        [71, 71], [72, 72], [73, 73], [74, 74], [75, 75], [76, 76],
        [77, 77], [78, 78], [79, 79], [80, 80], [81, 81], [82, 82],
        [83, 83], [84, 84], [85, 85], [86, 86], [87, 87], [88, 88],
        [89, 89], [90, 90], [91, 91], [92, 92], [93, 93], [94, 94],
        [95, 95], [96, 96], [97, 97], [98, 98], [99, 99], [100, 100]
    ];
    
    private $vpnRanges = [
        // Rangos conocidos de VPNs
        [103, 21], [103, 22], [103, 23], [103, 24], [103, 25],
        [104, 16], [104, 17], [104, 18], [104, 19], [104, 20],
        [107, 20], [107, 21], [107, 22], [107, 23], [107, 24],
        [108, 61], [108, 62], [108, 63], [108, 64], [108, 65],
        [109, 70], [109, 71], [109, 72], [109, 73], [109, 74],
        [110, 34], [110, 35], [110, 36], [110, 37], [110, 38],
        [111, 221], [111, 222], [111, 223], [111, 224], [111, 225],
        [112, 111], [112, 112], [112, 113], [112, 114], [112, 115],
        [113, 11], [113, 12], [113, 13], [113, 14], [113, 15],
        [114, 141], [114, 142], [114, 143], [114, 144], [114, 145],
        [115, 159], [115, 160], [115, 161], [115, 162], [115, 163]
    ];
    
    /**
     * Obtiene la IP real del cliente
     */
    public function getIP()
    {
        $ip = "";
        
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
        } else {
            $ip = $_SERVER['REMOTE_ADDR'];
        }
        
        // Manejar IP local
        if ($ip == "::1" || $ip == "127.0.0.1") {
            $ip = "146.70.230.150"; // IP de prueba
        }
        
        return $ip;
    }
    
    /**
     * Verifica si una IP es sospechosa sin llamadas externas
     */
    public function isSuspiciousIP($ip)
    {
        if (empty($ip)) {
            return false;
        }
        
        // Verificar si es una IP válida
        if (!filter_var($ip, FILTER_VALIDATE_IP)) {
            return true; // IP inválida es sospechosa
        }
        
        // Verificar si es IP privada o local
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
            // IP pública - analizar patrones
            return $this->analyzePublicIP($ip);
        }
        
        return false; // IP privada/local no es sospechosa
    }
    
    /**
     * Analiza una IP pública para detectar patrones sospechosos
     */
    private function analyzePublicIP($ip)
    {
        $ipParts = explode('.', $ip);
        
        if (count($ipParts) !== 4) {
            return true; // Formato inválido
        }
        
        $firstOctet = (int)$ipParts[0];
        $secondOctet = (int)$ipParts[1];
        $thirdOctet = (int)$ipParts[2];
        $fourthOctet = (int)$ipParts[3];
        
        // Verificar rangos de datacenters
        foreach ($this->suspiciousRanges as $range) {
            if ($firstOctet == $range[0] && $secondOctet == $range[1]) {
                return true;
            }
        }
        
        // Verificar rangos de VPNs
        foreach ($this->vpnRanges as $range) {
            if ($firstOctet == $range[0] && $secondOctet == $range[1]) {
                return true;
            }
        }
        
        // Verificar patrones específicos
        if ($this->hasSuspiciousPattern($firstOctet, $secondOctet, $thirdOctet, $fourthOctet)) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Verifica patrones específicos sospechosos
     */
    private function hasSuspiciousPattern($first, $second, $third, $fourth)
    {
        // Patrones conocidos de bots
        $suspiciousPatterns = [
            // Secuencias consecutivas
            [$first, $second, $third, $fourth] === [1, 1, 1, 1],
            [$first, $second, $third, $fourth] === [2, 2, 2, 2],
            [$first, $second, $third, $fourth] === [3, 3, 3, 3],
            
            // Patrones de datacenters
            $first >= 1 && $first <= 100 && $second >= 1 && $second <= 100,
            
            // IPs con muchos ceros
            $fourth == 0 || $third == 0,
            
            // Rangos específicos de proveedores de hosting
            ($first == 8 && $second >= 8 && $second <= 15),
            ($first == 9 && $second >= 9 && $second <= 15),
            ($first == 10 && $second >= 10 && $second <= 15),
            
            // Rangos de cloud providers
            ($first == 13 && $second >= 107 && $second <= 115),
            ($first == 18 && $second >= 206 && $second <= 220),
            ($first == 34 && $second >= 102 && $second <= 120),
            ($first == 35 && $second >= 184 && $second <= 200),
            ($first == 52 && $second >= 0 && $second <= 50),
            ($first == 54 && $second >= 144 && $second <= 160),
            ($first == 104 && $second >= 196 && $second <= 200),
            ($first == 107 && $second >= 20 && $second <= 25),
            ($first == 130 && $second >= 211 && $second <= 220),
            ($first == 142 && $second >= 250 && $second <= 255),
            ($first == 144 && $second >= 202 && $second <= 210),
            ($first == 146 && $second >= 101 && $second <= 110),
            ($first == 147 && $second >= 75 && $second <= 85),
            ($first == 157 && $second >= 240 && $second <= 255),
            ($first == 158 && $second >= 101 && $second <= 110),
            ($first == 159 && $second >= 203 && $second <= 210),
            ($first == 162 && $second >= 216 && $second <= 220),
            ($first == 172 && $second >= 16 && $second <= 31),
            ($first == 192 && $second >= 168 && $second <= 168),
            ($first == 198 && $second >= 51 && $second <= 100),
            ($first == 203 && $second >= 208 && $second <= 210),
            ($first == 204 && $second >= 236 && $second <= 240),
            ($first == 205 && $second >= 251 && $second <= 255),
            ($first == 207 && $second >= 46 && $second <= 50),
            ($first == 208 && $second >= 80 && $second <= 90),
            ($first == 209 && $second >= 85 && $second <= 95),
            ($first == 216 && $second >= 58 && $second <= 70),
            ($first == 216 && $second >= 200 && $second <= 210),
            ($first == 216 && $second >= 218 && $second <= 220),
            ($first == 216 && $second >= 239 && $second <= 255)
        ];
        
        foreach ($suspiciousPatterns as $pattern) {
            if ($pattern) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Verifica si la IP está en una lista negra local
     */
    public function isBlacklisted($ip)
    {
        $paths = require __DIR__ . '/paths.php';
        $blacklistFile = $paths['files']['blacklist'];
        
        if (!file_exists($blacklistFile)) {
            return false;
        }
        
        $blacklist = file($blacklistFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        
        return in_array($ip, $blacklist);
    }
    
    /**
     * Agrega una IP a la lista negra
     */
    public function addToBlacklist($ip)
    {
        $paths = require __DIR__ . '/paths.php';
        $blacklistFile = $paths['files']['blacklist'];
        $dataDir = dirname($blacklistFile);
        
        if (!is_dir($dataDir)) {
            mkdir($dataDir, $paths['permissions']['directory'], true);
        }
        
        if (!in_array($ip, $this->getBlacklist())) {
            file_put_contents($blacklistFile, $ip . PHP_EOL, FILE_APPEND | LOCK_EX);
        }
        
        return true;
    }
    
    /**
     * Obtiene la lista negra de IPs
     */
    private function getBlacklist()
    {
        $paths = require __DIR__ . '/paths.php';
        $blacklistFile = $paths['files']['blacklist'];
        
        if (!file_exists($blacklistFile)) {
            return [];
        }
        
        return file($blacklistFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    }
    

}
?>
