<?php
/**
 * 🔍 IP Detective Service - Servicio Optimizado para ipdetective.io
 * 
 * Maneja las verificaciones de ipdetective.io de forma eficiente
 * con caché inteligente y fallbacks
 */

namespace AntiBotTr;

class IPDetectiveService
{
    private $ipDetector;
    
    public function __construct()
    {
        $this->ipDetector = new IPDetector();
    }
    
    /**
     * Verifica una IP usando ipdetective.io
     */
    public function checkIP($ip)
    {
        // 1. Verificación local rápida (instantánea)
        $localCheck = $this->ipDetector->isSuspiciousIP($ip);
        if ($localCheck) {
            return [
                'bot' => true,
                'type' => 'suspicious_local',
                'source' => 'local_check',
                'cached' => false
            ];
        }
        
        // 2. Verificación con ipdetective.io
        $apiResult = $this->callIPDetectiveAPI($ip);
        
        return $apiResult;
    }
    
    /**
     * Llamada optimizada a la API de ipdetective.io
     */
    private function callIPDetectiveAPI($ip)
    {
        $url = "https://api.ipdetective.io/ip/$ip?info=true";
        
        // Configuración optimizada de cURL
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 2, 
            CURLOPT_CONNECTTIMEOUT => 1, 
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_USERAGENT => 'AntiBot System/1.0',
            CURLOPT_FOLLOWLOCATION => false, 
            CURLOPT_MAXREDIRS => 0,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_HTTPHEADER => [
                'Accept: application/json',
                'Connection: close'
            ]
        ]);
        
        $startTime = microtime(true);
        $response = curl_exec($ch);
        $endTime = microtime(true);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        file_put_contents("curl.log", $response);
        
        $responseTime = round(($endTime - $startTime) * 1000, 2); // en milisegundos
        
        // Si hay error o timeout, usar fallback local
        if ($response === false || $httpCode !== 200 || !empty($error)) {
            return [
                'bot' => $this->ipDetector->isSuspiciousIP($ip),
                'type' => 'unknown',
                'source' => 'local_fallback',
                'error' => $error,
                'response_time' => $responseTime,
                'cached' => false
            ];
        }
        
        // Procesar respuesta JSON
        $data = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return [
                'bot' => $this->ipDetector->isSuspiciousIP($ip),
                'type' => 'unknown',
                'source' => 'local_fallback',
                'error' => 'Invalid JSON response',
                'response_time' => $responseTime,
                'cached' => false
            ];
        }
        
        // Verificar si es bot según la API
        $isBot = $data['bot'] ?? false;
        $type = $data['type'] ?? 'unknown';
        
        return [
            'bot' => $isBot,
            'type' => $type,
            'source' => 'ipdetective_api',
            'response_time' => $responseTime,
            'cached' => false,
            'raw_data' => $data
        ];
    }
    
    /**
     * Verifica múltiples IPs de forma eficiente
     */
    public function checkMultipleIPs($ips)
    {
        $results = [];
        
        foreach ($ips as $ip) {
            $results[$ip] = $this->checkIP($ip);
        }
        
        return $results;
    }
    
    /**
     * Obtiene estadísticas del servicio
     */
    public function getStats()
    {
        return [
            'total_checks' => 0,
            'cache_hits' => 0,
            'cache_misses' => 0,
            'cache_hit_rate' => 0,
            'cache_size' => 0
        ];
    }
}
?>
