<?php
/**
 * 🗄️ Database Manager - Gestor de Base de Datos
 * 
 * Clase para manejar todas las operaciones de base de datos
 * de manera robusta, escalable y con prevención de duplicados
 */

namespace AntiBotTr;

class DatabaseManager
{
    private $conn;
    private $config;
    private $logger;
    
    public function __construct($connection = null)
    {
        $this->conn = $connection;
        $this->config = new ConfigManager();
        $this->logger = new Logger();
        
        if (!$this->conn) {
            $this->initializeConnection();
        }
    }
    
    /**
     * Inicializa la conexión a la base de datos
     */
    private function initializeConnection()
    {
        try {
            // Obtener configuración de base de datos
            $dbConfig = $this->config->getDatabaseConfig();
            
            $this->conn = new \mysqli(
                $dbConfig['host'],
                $dbConfig['username'],
                $dbConfig['password'],
                $dbConfig['database']
            );
            
            if ($this->conn->connect_error) {
                throw new \Exception("Error de conexión: " . $this->conn->connect_error);
            }
            
            $this->conn->set_charset("utf8mb4");
            
        } catch (\Exception $e) {
            $this->logger->logError("Error al conectar a la base de datos: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Verifica si una IP está en blacklist
     */
    public function isInBlacklist($ip)
    {
        try {
            $stmt = $this->conn->prepare("
                SELECT id, reason, expires_at, is_active 
                FROM antibot_blacklist 
                WHERE ip_address = ? AND is_active = 1
                LIMIT 1
            ");
            
            $stmt->bind_param("s", $ip);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows > 0) {
                $row = $result->fetch_assoc();
                
                // Verificar si ha expirado
                if ($row['expires_at'] && strtotime($row['expires_at']) < time()) {
                    // Marcar como inactiva
                    $this->deactivateBlacklistEntry($row['id']);
                    return false;
                }
                
                return true;
            }
            
            return false;
            
        } catch (\Exception $e) {
            $this->logger->logError("Error al verificar blacklist: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Agrega una IP a la blacklist (sin duplicados)
     */
    public function addToBlacklist($ip, $reason = '', $userAgent = '', $geoData = [], $userId = null, $userEmail = null)
    {
        try {
            // Verificar si ya existe
            if ($this->isInBlacklist($ip)) {
                // Incrementar contador de bloqueos
                $this->incrementBlacklistCount($ip);
                return false;
            }
            
            // Obtener configuración de expiración
            $expirationDays = (int)$this->config->get('blacklist_expiration_days', 30);
            $expiresAt = date('Y-m-d H:i:s', time() + ($expirationDays * 86400));
            
            $stmt = $this->conn->prepare("
                INSERT INTO antibot_blacklist 
                (ip_address, reason, user_agent, country_code, country_name, city, isp, expires_at, user_id, user_email) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE 
                reason = VALUES(reason),
                user_agent = VALUES(user_agent),
                country_code = VALUES(country_code),
                country_name = VALUES(country_name),
                city = VALUES(city),
                isp = VALUES(isp),
                expires_at = VALUES(expires_at),
                user_id = VALUES(user_id),
                user_email = VALUES(user_email),
                is_active = 1,
                block_count = block_count + 1,
                last_blocked_at = NOW()
            ");
            
            $countryCode = $geoData['country_code'] ?? null;
            $countryName = $geoData['country_name'] ?? null;
            $city = $geoData['city'] ?? null;
            $isp = $geoData['isp'] ?? null;
            
            $stmt->bind_param("ssssssssss", 
                $ip, $reason, $userAgent, $countryCode, $countryName, $city, $isp, $expiresAt, $userId, $userEmail
            );
            
            $result = $stmt->execute();
            
            if ($result) {
                // Registrar en logs
                $this->logAction($ip, 'blocked', $reason, $userAgent, $geoData, $userId, $userEmail);
                
                // Limpiar caché relacionado
                $this->clearIPCache($ip);
                
                return true;
            }
            
            return false;
            
        } catch (\Exception $e) {
            $this->logger->logError("Error al agregar a blacklist: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Verifica si una IP está en whitelist
     */
    public function isWhitelisted($ip)
    {
        try {
            $stmt = $this->conn->prepare("
                SELECT id, reason, expires_at, is_active 
                FROM antibot_whitelist 
                WHERE ip_address = ? AND is_active = 1
                LIMIT 1
            ");
            
            $stmt->bind_param("s", $ip);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows > 0) {
                $row = $result->fetch_assoc();
                
                // Verificar si ha expirado
                if ($row['expires_at'] && strtotime($row['expires_at']) < time()) {
                    // Marcar como inactiva
                    $this->deactivateWhitelistEntry($row['id']);
                    return false;
                }
                
                // Incrementar contador de accesos
                $this->incrementWhitelistAccess($ip);
                
                return true;
            }
            
            return false;
            
        } catch (\Exception $e) {
            $this->logger->logError("Error al verificar whitelist: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Agrega una IP a la whitelist (sin duplicados)
     */
    public function addToWhitelist($ip, $duration = 3600, $reason = '', $userAgent = '', $geoData = [], $userId = null, $userEmail = null)
    {
        try {
            // Verificar si ya existe
            if ($this->isWhitelisted($ip)) {
                // Extender duración si es necesario
                $this->extendWhitelistDuration($ip, $duration);
                return false;
            }
            
            $expiresAt = date('Y-m-d H:i:s', time() + $duration);
            
            $stmt = $this->conn->prepare("
                INSERT INTO antibot_whitelist 
                (ip_address, reason, user_agent, country_code, country_name, city, isp, expires_at, user_id, user_email) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE 
                reason = VALUES(reason),
                user_agent = VALUES(user_agent),
                country_code = VALUES(country_code),
                country_name = VALUES(country_name),
                city = VALUES(city),
                isp = VALUES(isp),
                expires_at = VALUES(expires_at),
                user_id = VALUES(user_id),
                user_email = VALUES(user_email),
                is_active = 1,
                access_count = access_count + 1,
                last_access_at = NOW()
            ");
            
            $countryCode = $geoData['country_code'] ?? null;
            $countryName = $geoData['country_name'] ?? null;
            $city = $geoData['city'] ?? null;
            $isp = $geoData['isp'] ?? null;
            
            $stmt->bind_param("ssssssssss", 
                $ip, $reason, $userAgent, $countryCode, $countryName, $city, $isp, $expiresAt, $userId, $userEmail
            );
            
            $result = $stmt->execute();
            
            if ($result) {
                // Registrar en logs
                $this->logAction($ip, 'whitelisted', $reason, $userAgent, $geoData, $userId, $userEmail);
                
                // Limpiar caché relacionado
                $this->clearIPCache($ip);
                
                return true;
            }
            
            return false;
            
        } catch (\Exception $e) {
            $this->logger->logError("Error al agregar a whitelist: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Remueve una IP de la blacklist
     */
    public function removeFromBlacklist($ip)
    {
        try {
            $stmt = $this->conn->prepare("
                UPDATE antibot_blacklist 
                SET is_active = 0 
                WHERE ip_address = ? AND is_active = 1
            ");
            
            $stmt->bind_param("s", $ip);
            $result = $stmt->execute();
            
            if ($result && $stmt->affected_rows > 0) {
                $this->logAction($ip, 'removed', 'Removido de blacklist', '', [], null, null);
                $this->clearIPCache($ip);
                return true;
            }
            
            return false;
            
        } catch (\Exception $e) {
            $this->logger->logError("Error al remover de blacklist: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Remueve una IP de la whitelist
     */
    public function removeFromWhitelist($ip)
    {
        try {
            $stmt = $this->conn->prepare("
                UPDATE antibot_whitelist 
                SET is_active = 0 
                WHERE ip_address = ? AND is_active = 1
            ");
            
            $stmt->bind_param("s", $ip);
            $result = $stmt->execute();
            
            if ($result && $stmt->affected_rows > 0) {
                $this->logAction($ip, 'removed', 'Removido de whitelist', '', [], null, null);
                $this->clearIPCache($ip);
                return true;
            }
            
            return false;
            
        } catch (\Exception $e) {
            $this->logger->logError("Error al remover de whitelist: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Obtiene estadísticas de blacklist
     */
    public function getBlacklistStats()
    {
        try {
            $stats = [];
            
            // Total activas
            $result = $this->conn->query("
                SELECT COUNT(*) as total_active 
                FROM antibot_blacklist 
                WHERE is_active = 1
            ");
            $stats['total_active'] = $result->fetch_assoc()['total_active'];
            
            // Total expiradas
            $result = $this->conn->query("
                SELECT COUNT(*) as total_expired 
                FROM antibot_blacklist 
                WHERE expires_at < NOW() AND is_active = 1
            ");
            $stats['total_expired'] = $result->fetch_assoc()['total_expired'];
            
            // Por país
            $result = $this->conn->query("
                SELECT country_code, COUNT(*) as count 
                FROM antibot_blacklist 
                WHERE is_active = 1 AND country_code IS NOT NULL 
                GROUP BY country_code 
                ORDER BY count DESC 
                LIMIT 10
            ");
            $stats['by_country'] = $result->fetch_all(MYSQLI_ASSOC);
            
            return $stats;
            
        } catch (\Exception $e) {
            $this->logger->logError("Error al obtener estadísticas de blacklist: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Obtiene estadísticas de whitelist
     */
    public function getWhitelistStats()
    {
        try {
            $stats = [];
            
            // Total activas
            $result = $this->conn->query("
                SELECT COUNT(*) as total_active 
                FROM antibot_whitelist 
                WHERE is_active = 1
            ");
            $stats['total_active'] = $result->fetch_assoc()['total_active'];
            
            // Total expiradas
            $result = $this->conn->query("
                SELECT COUNT(*) as total_expired 
                FROM antibot_whitelist 
                WHERE expires_at < NOW() AND is_active = 1
            ");
            $stats['total_expired'] = $result->fetch_assoc()['total_expired'];
            
            // Por país
            $result = $this->conn->query("
                SELECT country_code, COUNT(*) as count 
                FROM antibot_whitelist 
                WHERE is_active = 1 AND country_code IS NOT NULL 
                GROUP BY country_code 
                ORDER BY count DESC 
                LIMIT 10
            ");
            $stats['by_country'] = $result->fetch_all(MYSQLI_ASSOC);
            
            return $stats;
            
        } catch (\Exception $e) {
            $this->logger->logError("Error al obtener estadísticas de whitelist: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Ejecuta limpieza automática
     */
    public function runCleanup()
    {
        try {
            // Verificar si la limpieza automática está habilitada
            if (!$this->config->get('enable_auto_cleanup', 1)) {
                return ['blacklist_cleaned' => 0, 'whitelist_cleaned' => 0];
            }
            
            // Limpiar blacklist expirada
            $stmt = $this->conn->prepare("
                UPDATE antibot_blacklist 
                SET is_active = 0 
                WHERE expires_at < NOW() AND is_active = 1
            ");
            $stmt->execute();
            $blacklistCleaned = $stmt->affected_rows;
            
            // Limpiar whitelist expirada
            $stmt = $this->conn->prepare("
                UPDATE antibot_whitelist 
                SET is_active = 0 
                WHERE expires_at < NOW() AND is_active = 1
            ");
            $stmt->execute();
            $whitelistCleaned = $stmt->affected_rows;
            
            // Registrar limpieza
            if ($blacklistCleaned > 0 || $whitelistCleaned > 0) {
                $this->logAction('SYSTEM', 'removed', 
                    "Limpieza automática: {$blacklistCleaned} blacklist, {$whitelistCleaned} whitelist",
                    '', [], null, null
                );
            }
            
            return [
                'blacklist_cleaned' => $blacklistCleaned,
                'whitelist_cleaned' => $whitelistCleaned
            ];
            
        } catch (\Exception $e) {
            $this->logger->logError("Error en limpieza automática: " . $e->getMessage());
            return ['blacklist_cleaned' => 0, 'whitelist_cleaned' => 0];
        }
    }
    
    /**
     * Métodos privados auxiliares
     */
    private function incrementBlacklistCount($ip)
    {
        $stmt = $this->conn->prepare("
            UPDATE antibot_blacklist 
            SET block_count = block_count + 1, last_blocked_at = NOW() 
            WHERE ip_address = ?
        ");
        $stmt->bind_param("s", $ip);
        $stmt->execute();
    }
    
    private function incrementWhitelistAccess($ip)
    {
        $stmt = $this->conn->prepare("
            UPDATE antibot_whitelist 
            SET access_count = access_count + 1, last_access_at = NOW() 
            WHERE ip_address = ?
        ");
        $stmt->bind_param("s", $ip);
        $stmt->execute();
    }
    
    private function extendWhitelistDuration($ip, $duration)
    {
        $expiresAt = date('Y-m-d H:i:s', time() + $duration);
        $stmt = $this->conn->prepare("
            UPDATE antibot_whitelist 
            SET expires_at = ?, access_count = access_count + 1, last_access_at = NOW() 
            WHERE ip_address = ?
        ");
        $stmt->bind_param("ss", $expiresAt, $ip);
        $stmt->execute();
    }
    
    private function deactivateBlacklistEntry($id)
    {
        $stmt = $this->conn->prepare("UPDATE antibot_blacklist SET is_active = 0 WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
    }
    
    private function deactivateWhitelistEntry($id)
    {
        $stmt = $this->conn->prepare("UPDATE antibot_whitelist SET is_active = 0 WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
    }
    
    public function logAction($ip, $action, $reason = '', $userAgent = '', $geoData = [], $userId = null, $userEmail = null)
    {
        try {
            $stmt = $this->conn->prepare("
                INSERT INTO antibot_logs 
                (ip_address, action, reason, user_agent, url, country_code, country_name, city, isp, user_id, user_email) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            $url = $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'] ?? '';
            $countryCode = $geoData['country_code'] ?? null;
            $countryName = $geoData['country_name'] ?? null;
            $city = $geoData['city'] ?? null;
            $isp = $geoData['isp'] ?? null;
            
            $stmt->bind_param("sssssssssss", 
                $ip, $action, $reason, $userAgent, $url, $countryCode, $countryName, $city, $isp, $userId, $userEmail
            );
            
            $stmt->execute();
            
        } catch (\Exception $e) {
            $this->logger->logError("Error al registrar log: " . $e->getMessage());
        }
    }
    
    private function clearIPCache($ip)
    {

    }
    
    /**
     * Obtiene la conexión a la base de datos
     */
    public function getConnection()
    {
        return $this->conn;
    }
    
    /**
     * Cierra la conexión
     */
    public function close()
    {
        if ($this->conn) {
            $this->conn->close();
        }
    }
    
    /**
     * Destructor
     */
    public function __destruct()
    {
        $this->close();
    }
}
?>
