<?php
/**
 * ⚙️ Config Manager - Gestor de Configuración
 * 
 * Clase para manejar la configuración del sistema desde la base de datos
 */

namespace AntiBotTr;

class ConfigManager
{
    private $conn;
    private $configCache = [];
    
    public function __construct($connection = null)
    {
        $this->conn = $connection;
        
        if (!$this->conn) {
            $this->initializeConnection();
        }
    }
    
    /**
     * Inicializa la conexión a la base de datos
     */
    private function initializeConnection()
    {
        try {
            // Leer configuración desde mysql.php
            $dbConfig = $this->loadDatabaseConfig();
            
            $this->conn = new \mysqli(
                $dbConfig['host'],
                $dbConfig['username'],
                $dbConfig['password'],
                $dbConfig['database']
            );
            
            if ($this->conn->connect_error) {
                throw new \Exception("Error de conexión: " . $this->conn->connect_error);
            }
            
            $this->conn->set_charset("utf8mb4");
            
        } catch (\Exception $e) {
            // Si no hay conexión, usar configuración por defecto
            $this->configCache = $this->getDefaultConfig();
        }
    }
    
    /**
     * Carga la configuración de base de datos desde mysql.php
     */
    private function loadDatabaseConfig()
    {
        $mysqlFile = __DIR__ . '/../php/mysql.php';
        
        if (!file_exists($mysqlFile)) {
            throw new \Exception("Archivo mysql.php no encontrado en: " . $mysqlFile);
        }
        
        // Incluir el archivo mysql.php para obtener las variables
        include $mysqlFile;
        
        // Verificar que las variables estén definidas
        if (!isset($servername) || !isset($username) || !isset($password) || !isset($dbname)) {
            throw new \Exception("Variables de configuración de base de datos no encontradas en mysql.php");
        }
        
        return [
            'host' => $servername,
            'username' => $username,
            'password' => $password,
            'database' => $dbname
        ];
    }
    
    /**
     * Obtiene un valor de configuración
     */
    public function get($key, $default = null)
    {
        // Verificar caché primero
        if (isset($this->configCache[$key])) {
            return $this->configCache[$key];
        }
        

        
        try {
            if (!$this->conn) {
                return $this->getDefaultValue($key, $default);
            }
            
            $stmt = $this->conn->prepare("
                SELECT key_value FROM antibot_config 
                WHERE key_name = ? 
                LIMIT 1
            ");
            
            $stmt->bind_param("s", $key);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows > 0) {
                $value = $result->fetch_assoc()['key_value'];
                $this->configCache[$key] = $value;
                return $value;
            }
            
            return $this->getDefaultValue($key, $default);
            
        } catch (\Exception $e) {
            return $this->getDefaultValue($key, $default);
        }
    }
    
    /**
     * Establece un valor de configuración
     */
    public function set($key, $value, $description = '')
    {
        try {
            if (!$this->conn) {
                $this->configCache[$key] = $value;
                return true;
            }
            
            $stmt = $this->conn->prepare("
                INSERT INTO antibot_config (key_name, key_value, description) 
                VALUES (?, ?, ?) 
                ON DUPLICATE KEY UPDATE 
                key_value = VALUES(key_value),
                description = VALUES(description),
                updated_at = NOW()
            ");
            
            $stmt->bind_param("sss", $key, $value, $description);
            $result = $stmt->execute();
            
            if ($result) {
                $this->configCache[$key] = $value;
                return true;
            }
            
            return false;
            
        } catch (\Exception $e) {
            $this->configCache[$key] = $value;
            return true;
        }
    }
    
    /**
     * Obtiene toda la configuración
     */
    public function getAll()
    {
        try {
            if (!$this->conn) {
                return $this->getDefaultConfig();
            }
            
            $result = $this->conn->query("
                SELECT key_name, key_value, description 
                FROM antibot_config 
                ORDER BY key_name
            ");
            
            $config = [];
            while ($row = $result->fetch_assoc()) {
                $config[$row['key_name']] = $row['key_value'];
            }
            
            return array_merge($this->getDefaultConfig(), $config);
            
        } catch (\Exception $e) {
            return $this->getDefaultConfig();
        }
    }
    
    /**
     * Obtiene la configuración de base de datos desde mysql.php
     */
    public function getDatabaseConfig()
    {
        return $this->loadDatabaseConfig();
    }
    
    /**
     * Obtiene configuración por defecto
     */
    private function getDefaultConfig()
    {
        return [
            'auto_whitelist_duration' => '3600',
            'blacklist_expiration_days' => '30',
            'whitelist_expiration_days' => '7',
            'max_blacklist_entries' => '10000',
            'max_whitelist_entries' => '5000',
            'cleanup_interval_hours' => '24',
            'enable_auto_cleanup' => '1',
            'enable_telegram_notifications' => '1',
            'telegram_bot_token' => '',
            'telegram_chat_id' => '',
            'enable_geo_blocking' => '1',
            'blocked_countries' => 'MX,US',
            'db_host' => 'localhost',
            'db_username' => 'root',
            'db_password' => '',
            'db_database' => 'antibot_db'
        ];
    }
    
    /**
     * Obtiene valor por defecto para una clave específica
     */
    private function getDefaultValue($key, $default = null)
    {
        $defaultConfig = $this->getDefaultConfig();
        
        if (isset($defaultConfig[$key])) {
            return $defaultConfig[$key];
        }
        
        return $default;
    }
    
    /**
     * Limpia el caché de configuración
     */
    public function clearCache()
    {
        $this->configCache = [];
    }
    
    /**
     * Verifica si una configuración existe
     */
    public function has($key)
    {
        return $this->get($key) !== null;
    }
    
    /**
     * Elimina una configuración
     */
    public function delete($key)
    {
        try {
            if (!$this->conn) {
                unset($this->configCache[$key]);
                return true;
            }
            
            $stmt = $this->conn->prepare("DELETE FROM antibot_config WHERE key_name = ?");
            $stmt->bind_param("s", $key);
            $result = $stmt->execute();
            
            if ($result) {
                unset($this->configCache[$key]);
                return true;
            }
            
            return false;
            
        } catch (\Exception $e) {
            unset($this->configCache[$key]);
            return true;
        }
    }
    
    /**
     * Obtiene configuración de Telegram
     */
    public function getTelegramConfig()
    {
        return [
            'enabled' => (bool)$this->get('enable_telegram_notifications', 1),
            'bot_token' => $this->get('telegram_bot_token', ''),
            'chat_id' => $this->get('telegram_chat_id', '')
        ];
    }
    
    /**
     * Obtiene configuración de geolocalización
     */
    public function getGeoConfig()
    {
        return [
            'enabled' => (bool)$this->get('enable_geo_blocking', 1),
            'blocked_countries' => explode(',', $this->get('blocked_countries', 'MX,US'))
        ];
    }
    
    /**
     * Cierra la conexión
     */
    public function close()
    {
        if ($this->conn) {
            $this->conn->close();
        }
    }
    
    /**
     * Destructor
     */
    public function __destruct()
    {
        $this->close();
    }
}
?>
